/*
 * Written by Dawid Kurzyniec and released to the public domain, as explained
 * at http://creativecommons.org/licenses/publicdomain
 */

package edu.emory.mathcs.util.collections;

import java.util.*;
import edu.emory.mathcs.backport.java.util.concurrent.*;

/**
 * Hash map that refers to values via weak references, thus not
 * keeping them from garbage collection, asynchronously removing entries
 * which values have been unreferenced and garbage collected. This class
 * is a companion to {@link java.util.WeakHashMap}.
 *
 * @author Dawid Kurzyniec
 * @version 1.0
 */
public class AsyncWeakValueHashMap extends AsyncWeakValueMap {

    /**
     * Creates new AsyncWeakValueHashMap with specified initial capacity and load
     * factor.
     *
     * @see java.util.HashMap(int,float)
     */
    public AsyncWeakValueHashMap(int initialCapacity, float loadFactor) {
        super(new ConcurrentHashMap(initialCapacity, loadFactor));
    }

    /**
     * Creates new AsyncWeakValueHashMap with specified initial capacity and a
     * default load factor.
     *
     * @see java.util.HashMap(int)
     */
    public AsyncWeakValueHashMap(int initialCapacity) {
        super(new ConcurrentHashMap(initialCapacity));
    }

    /**
     * Creates new AsyncWeakValueHashMap with a default initial capacity and a
     * default load factor.
     *
     * @see java.util.HashMap(int)
     */
    public AsyncWeakValueHashMap() {
        super(new ConcurrentHashMap());
    }

    /**
     * Creates new AsyncWeakValueHashMap with specified initial capacity and a
     * default load factor, and copies to it all the mappings from the
     * specified map.
     *
     * @see java.util.HashMap(int)
     */
    public AsyncWeakValueHashMap(Map t) {
        super (new ConcurrentHashMap());
        putAll(t);
    }
}
